<?php
// QR Kod Servis Görüntüleme Sayfası (Müşteri View)
// URL: /servis/?h=<qr_hash>

declare(strict_types=1);

function e(string $value): string
{
    return htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
}

function status_meta(string $status): array
{
    $map = [
        'completed' => ['Tamamlandı', 'ok'],
        'in_progress' => ['Devam Ediyor', 'warn'],
        'pending' => ['Beklemede', 'info'],
        'cancelled' => ['İptal', 'bad'],
    ];
    return $map[$status] ?? [$status, ''];
}

function item_type_text(string $type): string
{
    $map = [
        'labor' => 'İşçilik',
        'part' => 'Parça',
        'material' => 'Malzeme',
    ];
    return $map[$type] ?? $type;
}

function icon_svg(string $name): string
{
    $common = 'fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"';

    switch ($name) {
        case 'wrench':
            return '<svg width="18" height="18" viewBox="0 0 24 24" ' . $common . '><path d="M14.7 6.3a4 4 0 0 0-5.4 5.4L3 18l3 3 6.3-6.3a4 4 0 0 0 5.4-5.4l-2.2 2.2-2.4-2.4 2.2-2.2Z"/></svg>';
        case 'box':
            return '<svg width="18" height="18" viewBox="0 0 24 24" ' . $common . '><path d="M21 8l-9 5-9-5 9-5 9 5Z"/><path d="M3 8v8l9 5 9-5V8"/></svg>';
        default:
            return '<svg width="18" height="18" viewBox="0 0 24 24" ' . $common . '><path d="M9 5h6"/><path d="M9 3h6a2 2 0 0 1 2 2v16H7V5a2 2 0 0 1 2-2Z"/><path d="M9 9h6"/><path d="M9 13h6"/><path d="M9 17h6"/></svg>';
    }
}

function render_page(string $title, string $contentHtml): void
{
    ?>
    <!doctype html>
    <html lang="tr">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0, viewport-fit=cover">
        <meta name="theme-color" content="#E00000">
        <title><?php echo e($title); ?></title>
        <link rel="icon" href="../icon.png">
        <link rel="apple-touch-icon" href="../icon.png">
        <style>
            :root{
                --brand:#E00000;
                --brand-2:#B80000;
                --ink:#0B0B10;
                --bg:#F3F5F8;
                --card:#FFFFFF;
                --text:#111827;
                --muted:#6B7280;
                --line:#E6E9EF;
                --shadow:0 10px 24px rgba(16,24,40,.08);
                --radius:16px;
                --radius2:12px;
            }
            *{box-sizing:border-box}
            html,body{height:100%}
            body{margin:0;font-family:ui-sans-serif,system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;color:var(--text);background:var(--bg)}
            .safe{padding: env(safe-area-inset-top) 0 env(safe-area-inset-bottom)}

            .topbar{position:sticky;top:0;z-index:20;background:rgba(255,255,255,.92);backdrop-filter:blur(10px);border-bottom:1px solid var(--line)}
            .topbar::before{content:"";display:block;height:4px;background:linear-gradient(90deg,var(--brand),var(--brand-2))}
            .topbar-inner{max-width:1024px;margin:0 auto;padding:12px 16px;display:flex;align-items:center;justify-content:flex-start}
            .brand-only{display:flex;align-items:center;justify-content:center;width:100%}
            .brand-only img{height:34px;width:auto;max-width:200px;object-fit:contain}

            .wrap{max-width:1024px;margin:0 auto;padding:16px}
            .summary{border:1px solid var(--line);border-radius:var(--radius);background:var(--card);box-shadow:var(--shadow);padding:16px}
            .summary h1{margin:0;font-size:18px;letter-spacing:.2px}
            .summary p{margin:6px 0 0;color:var(--muted);font-size:13px;line-height:1.45}

            .grid{display:grid;grid-template-columns:1fr;gap:14px;margin-top:14px}
            @media (min-width: 880px){.grid{grid-template-columns: 1.1fr .9fr}}

            .card{border:1px solid var(--line);border-radius:var(--radius);background:var(--card);box-shadow:var(--shadow);overflow:hidden}
            .card-header{padding:14px 16px;border-bottom:1px solid var(--line);background:linear-gradient(180deg, rgba(224,0,0,.06), transparent 90%)}
            .card-header h2{margin:0;font-size:14px;letter-spacing:.2px;display:flex;align-items:center;gap:10px}
            .card-header h2::before{content:"";width:10px;height:10px;border-radius:4px;background:linear-gradient(180deg,var(--brand),var(--brand-2));box-shadow:0 8px 18px rgba(224,0,0,.14);display:inline-block}
            .card-body{padding:14px 16px}

            .kv{display:grid;grid-template-columns: 140px 1fr;gap:10px 12px}
            .k{color:var(--muted);font-size:12px}
            .v{font-size:14px}

            .empty{padding:18px 16px;color:var(--muted);text-align:center}

            .history{display:flex;flex-direction:column;gap:10px}
            details.service{border:1px solid var(--line);border-radius:var(--radius2);background:#fff;overflow:hidden}
            summary{list-style:none;cursor:pointer;padding:14px 14px;display:flex;align-items:flex-start;justify-content:space-between;gap:12px}
            summary::-webkit-details-marker{display:none}
            details.service[open] summary{border-bottom:1px solid var(--line);background:linear-gradient(90deg, rgba(224,0,0,.05), transparent 70%)}

            .sum-title{display:flex;align-items:center;gap:10px;flex-wrap:wrap}
            .date{font-weight:800}
            .chev{color:var(--muted);font-size:12px;align-self:center}

            .chip{display:inline-flex;align-items:center;gap:6px;font-size:12px;padding:6px 10px;border-radius:999px;border:1px solid var(--line);background:#fff;color:var(--text)}
            .chip.ok{background: rgba(34,197,94,.10);border-color: rgba(34,197,94,.20);color:#14532d}
            .chip.warn{background: rgba(245,158,11,.12);border-color: rgba(245,158,11,.22);color:#7a3e00}
            .chip.info{background: rgba(59,130,246,.10);border-color: rgba(59,130,246,.20);color:#0b3d73}
            .chip.bad{background: rgba(239,68,68,.10);border-color: rgba(239,68,68,.20);color:#7a1510}

            .service-body{padding:0 14px 14px}
            .desc{margin:10px 0 0;padding:12px 12px;border-radius:12px;border:1px solid var(--line);background:rgba(15,23,42,.02);line-height:1.45;white-space:pre-line}
            .items{margin-top:12px;border-top:1px solid var(--line);padding-top:12px;display:flex;flex-direction:column;gap:10px}
            .item{display:flex;gap:10px;align-items:flex-start;padding:10px 10px;border-radius:12px;border:1px solid var(--line);background:#fff}
            .i-ico{width:34px;height:34px;border-radius:10px;display:flex;align-items:center;justify-content:center;background:rgba(224,0,0,.06);border:1px solid rgba(224,0,0,.16);flex:0 0 auto;color:var(--brand)}
            .i-title{display:flex;gap:8px;align-items:center;flex-wrap:wrap}
            .i-title strong{font-size:13px}
            .qty{font-size:12px;color:var(--muted)}

            .footer{padding:22px 16px;color:var(--muted);text-align:center;font-size:12px}
        </style>
    </head>
    <body>
    <div class="safe">
        <?php echo $contentHtml; ?>
    </div>
    </body>
    </html>
    <?php
}

function render_simple_message(string $message): void
{
    ob_start();
    ?>
    <header class="topbar">
        <div class="topbar-inner">
            <div class="brand-only"><img src="../logo.png" alt="ABALI"></div>
        </div>
    </header>
    <main class="wrap">
        <section class="summary">
            <h1>ABALI</h1>
            <p><?php echo e($message); ?></p>
        </section>
        <div class="footer">&copy; <?php echo e((string)date('Y')); ?> ABALI</div>
    </main>
    <?php
    render_page('ABALI', ob_get_clean());
    exit();
}

$hash = isset($_GET['h']) ? trim((string)$_GET['h']) : '';
if ($hash === '') {
    render_simple_message('QR kod parametresi bulunamadı. Lütfen QR kodu tekrar tarayın.');
}

$api_url = 'https://abali.online/api/qr/scan.php?hash=' . urlencode($hash);
$context = stream_context_create([
    'http' => [
        'method' => 'GET',
        'timeout' => 8,
        'header' => "Accept: application/json\r\nUser-Agent: ABALI-Servis/1.0\r\n",
    ],
    'ssl' => [
        'verify_peer' => true,
        'verify_peer_name' => true,
    ],
]);

$response = @file_get_contents($api_url, false, $context);
if ($response === false) {
    render_simple_message('Bilgi alınamadı. Lütfen daha sonra tekrar deneyin.');
}

$data = json_decode($response, true);
if (!is_array($data)) {
    render_simple_message('Beklenmeyen sunucu yanıtı alındı. Lütfen daha sonra tekrar deneyin.');
}

if (!($data['success'] ?? false)) {
    $msg = (string)($data['message'] ?? 'Bir hata oluştu');
    render_simple_message($msg);
}

$vehicle = $data['vehicle'] ?? null;
$customer = $data['customer'] ?? null;
$service_history = $data['service_history'] ?? [];
$is_assigned = (bool)($data['qr_code']['is_assigned'] ?? false);

if (!$is_assigned || !is_array($vehicle)) {
    render_simple_message('Bu QR kod henüz bir araca atanmadı.');
}

ob_start();
?>
<header class="topbar">
    <div class="topbar-inner">
        <div class="brand-only"><img src="../logo.png" alt="ABALI"></div>
    </div>
</header>

<main class="wrap">
    <section class="summary">
       
        <h3>Aşağıdan araç bilgilerini ve servis geçmişini görüntüleyebilirsiniz.</h3>
    </section>

    <div class="grid">
        <section class="card">
            <div class="card-header"><h2>Araç Bilgileri</h2></div>
            <div class="card-body">
                <div class="kv">
                    <div class="k">Plaka</div>
                    <div class="v"><?php echo e((string)($vehicle['plate'] ?? '')); ?></div>
                    <div class="k">Marka / Model</div>
                    <div class="v"><?php echo e(trim((string)($vehicle['brand'] ?? '') . ' ' . (string)($vehicle['model'] ?? ''))); ?></div>
                    <?php if (!empty($vehicle['year'])): ?>
                        <div class="k">Yıl</div>
                        <div class="v"><?php echo e((string)$vehicle['year']); ?></div>
                    <?php endif; ?>
                    <?php if (!empty($vehicle['color'])): ?>
                        <div class="k">Renk</div>
                        <div class="v"><?php echo e((string)$vehicle['color']); ?></div>
                    <?php endif; ?>
                </div>
            </div>
        </section>

        <section class="card">
            <div class="card-header"><h2>Müşteri Bilgileri</h2></div>
            <div class="card-body">
                <div class="kv">
                    <div class="k">Ad Soyad</div>
                    <div class="v"><?php echo e((string)($customer['name'] ?? '')); ?></div>
                    <div class="k">Telefon</div>
                    <div class="v"><?php echo e((string)($customer['phone'] ?? '')); ?></div>
                </div>
            </div>
        </section>
    </div>

    <section class="card" style="margin-top:14px">
        <div class="card-header"><h2>Servis Geçmişi</h2></div>
        <div class="card-body">
            <?php if (!is_array($service_history) || count($service_history) === 0): ?>
                <div class="empty">Henüz servis kaydı bulunmuyor.</div>
            <?php else: ?>
                <div class="history">
                    <?php foreach ($service_history as $idx => $service): ?>
                        <?php
                        $date = !empty($service['service_date']) ? date('d.m.Y', strtotime((string)$service['service_date'])) : '-';
                        [$statusLabel, $tone] = status_meta((string)($service['status'] ?? ''));
                        $km = !empty($service['km_reading']) ? number_format((float)$service['km_reading'], 0, ',', '.') : null;
                        $desc = (string)($service['description'] ?? '');
                        $items = $service['items'] ?? [];
                        $open = $idx === 0 ? 'open' : '';
                        ?>
                        <details class="service" <?php echo $open; ?>>
                            <summary>
                                <div class="sum-title">
                                    <span class="date"><?php echo e($date); ?></span>
                                    <span class="chip <?php echo e($tone); ?>"><?php echo e($statusLabel); ?></span>
                                    <?php if ($km): ?>
                                        <span class="chip">Km: <?php echo e($km); ?></span>
                                    <?php endif; ?>
                                    <?php if (is_array($items) && count($items) > 0): ?>
                                        <span class="chip"><?php echo e((string)count($items)); ?> kalem</span>
                                    <?php endif; ?>
                                </div>
                                <div class="chev">Detay</div>
                            </summary>

                            <div class="service-body">
                                <?php if ($desc !== ''): ?>
                                    <div class="desc"><?php echo e($desc); ?></div>
                                <?php endif; ?>

                                <div class="items">
                                    <div class="k" style="font-size:12px;margin-bottom:2px">Yapılan işlemler</div>
                                    <?php if (!is_array($items) || count($items) === 0): ?>
                                        <div class="empty" style="padding:10px 0">Bu servis için kayıtlı işlem bulunmuyor.</div>
                                    <?php else: ?>
                                        <?php foreach ($items as $item): ?>
                                            <?php
                                            $type = (string)($item['item_type'] ?? '');
                                            $typeText = item_type_text($type);
                                            $qty = (string)($item['quantity'] ?? '1');
                                            $itemDesc = (string)($item['description'] ?? '');
                                            $svg = $type === 'labor' ? icon_svg('wrench') : ($type === 'material' ? icon_svg('box') : icon_svg('clipboard'));
                                            ?>
                                            <div class="item">
                                                <div class="i-ico"><?php echo $svg; ?></div>
                                                <div class="i-main">
                                                    <div class="i-title">
                                                        <strong><?php echo e($itemDesc); ?></strong>
                                                        <span class="chip"><?php echo e($typeText); ?></span>
                                                    </div>
                                                    <div class="qty">Adet: <?php echo e($qty); ?></div>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </details>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </section>

    <div class="footer">&copy; <?php echo e((string)date('Y')); ?> ABALI</div>
</main>
<?php
$body = ob_get_clean();
render_page('ABALI', $body);
