<?php
declare(strict_types=1);

require_once __DIR__ . '/_bootstrap.php';
require_once __DIR__ . '/_layout.php';

require_admin();

function post_str(string $key): string
{
    $v = $_POST[$key] ?? '';
    return is_string($v) ? trim($v) : '';
}

function post_int(string $key): int
{
    $v = $_POST[$key] ?? 0;
    if (is_int($v)) return $v;
    return (int)preg_replace('/\D+/', '', (string)$v);
}

function redirect_self(): void
{
    header('Location: users.php');
    exit();
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    csrf_require();
    $action = post_str('action');

    try {
        if ($action === 'create_user') {
            $username = post_str('username');
            $fullName = post_str('full_name');
            $role = post_str('role');
            $phone = post_str('phone');
            $email = post_str('email');
            $password = (string)($_POST['password'] ?? '');

            if ($username === '' || $fullName === '' || $password === '') {
                flash_set('bad', 'Kullanıcı adı, ad soyad ve şifre zorunlu.');
                redirect_self();
            }
            if (!in_array($role, ['admin', 'technician'], true)) {
                flash_set('bad', 'Geçersiz rol.');
                redirect_self();
            }
            if (strlen($password) < 6) {
                flash_set('bad', 'Şifre en az 6 karakter olmalı.');
                redirect_self();
            }

            $hash = password_hash($password, PASSWORD_BCRYPT);
            $stmt = db()->prepare(
                "INSERT INTO users (username, password, full_name, role, phone, email, is_active)
                 VALUES (:u, :p, :f, :r, :ph, :e, 1)"
            );
            $stmt->execute([
                ':u' => $username,
                ':p' => $hash,
                ':f' => $fullName,
                ':r' => $role,
                ':ph' => $phone !== '' ? $phone : null,
                ':e' => $email !== '' ? $email : null,
            ]);
            flash_set('ok', 'Kullanıcı eklendi.');
            redirect_self();
        }

        if ($action === 'change_password') {
            $userId = post_int('user_id');
            $password = (string)($_POST['new_password'] ?? '');
            $password2 = (string)($_POST['new_password2'] ?? '');

            if ($userId <= 0) {
                flash_set('bad', 'Kullanıcı seçin.');
                redirect_self();
            }
            if ($password === '' || $password2 === '' || $password !== $password2) {
                flash_set('bad', 'Şifreler uyuşmuyor.');
                redirect_self();
            }
            if (strlen($password) < 6) {
                flash_set('bad', 'Şifre en az 6 karakter olmalı.');
                redirect_self();
            }

            $hash = password_hash($password, PASSWORD_BCRYPT);
            $stmt = db()->prepare("UPDATE users SET password = :p WHERE id = :id");
            $stmt->execute([':p' => $hash, ':id' => $userId]);
            flash_set('ok', 'Şifre güncellendi.');
            redirect_self();
        }

        if ($action === 'toggle_active') {
            $userId = post_int('user_id');
            $active = post_int('is_active') ? 1 : 0;

            if ($userId <= 0) {
                flash_set('bad', 'Kullanıcı seçin.');
                redirect_self();
            }
            // Kendini pasifleştirmeyi engelle
            if ((int)($_SESSION['admin_user_id'] ?? 0) === $userId && $active === 0) {
                flash_set('warn', 'Kendi hesabınızı pasif yapamazsınız.');
                redirect_self();
            }

            $stmt = db()->prepare("UPDATE users SET is_active = :a WHERE id = :id");
            $stmt->execute([':a' => $active, ':id' => $userId]);
            flash_set('ok', $active ? 'Kullanıcı aktif edildi.' : 'Kullanıcı pasif edildi.');
            redirect_self();
        }

        if ($action === 'update_user') {
            $userId = post_int('user_id');
            $fullName = post_str('full_name');
            $role = post_str('role');
            $phone = post_str('phone');
            $email = post_str('email');

            if ($userId <= 0) {
                flash_set('bad', 'Kullanıcı seçin.');
                redirect_self();
            }
            if ($fullName === '') {
                flash_set('bad', 'Ad soyad boş olamaz.');
                redirect_self();
            }
            if (!in_array($role, ['admin', 'technician'], true)) {
                flash_set('bad', 'Geçersiz rol.');
                redirect_self();
            }

            // Kendini technician yapmayı engelle
            if ((int)($_SESSION['admin_user_id'] ?? 0) === $userId && $role !== 'admin') {
                flash_set('warn', 'Kendi rolünüzü admin dışına alamazsınız.');
                redirect_self();
            }

            $stmt = db()->prepare(
                "UPDATE users
                 SET full_name = :f, role = :r, phone = :ph, email = :e
                 WHERE id = :id"
            );
            $stmt->execute([
                ':f' => $fullName,
                ':r' => $role,
                ':ph' => $phone !== '' ? $phone : null,
                ':e' => $email !== '' ? $email : null,
                ':id' => $userId,
            ]);
            flash_set('ok', 'Kullanıcı güncellendi.');
            redirect_self();
        }

        flash_set('bad', 'Geçersiz işlem.');
        redirect_self();
    } catch (Throwable $e) {
        $msg = $e->getMessage();
        if (stripos($msg, 'Duplicate') !== false || stripos($msg, 'UNIQUE') !== false) {
            flash_set('bad', 'Bu kullanıcı adı zaten kayıtlı.');
            redirect_self();
        }
        flash_set('bad', 'Sunucu hatası.');
        redirect_self();
    }
}

$users = [];
try {
    $stmt = db()->query("SELECT id, username, full_name, role, phone, email, is_active, created_at FROM users ORDER BY created_at DESC, id DESC");
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
    $users = [];
}

ob_start();
?>
<div class="grid">
    <div class="card">
        <h2>Kullanıcılar</h2>
        <div class="body">
            <div class="muted" style="margin-bottom:10px">
                Admin tamamlayana kadar mobil uygulama bu kullanıcılarla giriş yapar.
            </div>
            <div style="overflow:auto">
                <table>
                    <thead>
                    <tr>
                        <th>ID</th>
                        <th>Kullanıcı</th>
                        <th>Ad Soyad</th>
                        <th>Rol</th>
                        <th>Durum</th>
                        <th>İletişim</th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($users as $u): ?>
                        <?php
                        $isActive = ((int)($u['is_active'] ?? 0)) === 1;
                        $role = (string)($u['role'] ?? '');
                        ?>
                        <tr>
                            <td><?php echo (int)($u['id'] ?? 0); ?></td>
                            <td><strong><?php echo e((string)($u['username'] ?? '')); ?></strong></td>
                            <td><?php echo e((string)($u['full_name'] ?? '')); ?></td>
                            <td>
                                <span class="tag <?php echo $role === 'admin' ? 'warn' : ''; ?>">
                                    <?php echo e($role === 'admin' ? 'admin' : 'technician'); ?>
                                </span>
                            </td>
                            <td>
                                <span class="tag <?php echo $isActive ? 'ok' : 'bad'; ?>">
                                    <?php echo $isActive ? 'aktif' : 'pasif'; ?>
                                </span>
                            </td>
                            <td class="muted">
                                <?php echo e((string)($u['phone'] ?? '')); ?>
                                <?php if (!empty($u['email'])): ?>
                                    <div><?php echo e((string)$u['email']); ?></div>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    <?php if (empty($users)): ?>
                        <tr><td colspan="6" class="muted">Kayıt bulunamadı.</td></tr>
                    <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <div>
        <div class="card" style="margin-bottom:16px">
            <h2>Yeni Kullanıcı</h2>
            <div class="body">
                <form method="post">
                    <input type="hidden" name="csrf" value="<?php echo e(csrf_token()); ?>">
                    <input type="hidden" name="action" value="create_user">

                    <label>Kullanıcı adı *</label>
                    <input name="username" required placeholder="örn: teknisyen1">

                    <label>Ad Soyad *</label>
                    <input name="full_name" required placeholder="örn: Ahmet Usta">

                    <div class="split">
                        <div>
                            <label>Rol</label>
                            <select name="role" required>
                                <option value="technician">technician</option>
                                <option value="admin">admin</option>
                            </select>
                        </div>
                        <div>
                            <label>Telefon</label>
                            <input name="phone" placeholder="örn: 05xx...">
                        </div>
                    </div>

                    <label>E-posta</label>
                    <input name="email" type="email" placeholder="örn: mail@...">

                    <label>Şifre *</label>
                    <input name="password" type="password" required placeholder="en az 6 karakter">

                    <div style="margin-top:14px" class="actions">
                        <button class="btn" type="submit">KULLANICI EKLE</button>
                    </div>
                </form>
            </div>
        </div>

        <div class="card" style="margin-bottom:16px">
            <h2>Şifre Değiştir</h2>
            <div class="body">
                <form method="post">
                    <input type="hidden" name="csrf" value="<?php echo e(csrf_token()); ?>">
                    <input type="hidden" name="action" value="change_password">

                    <label>Kullanıcı</label>
                    <select name="user_id" required>
                        <option value="">Seçin...</option>
                        <?php foreach ($users as $u): ?>
                            <option value="<?php echo (int)$u['id']; ?>">
                                <?php echo e((string)$u['username']); ?> — <?php echo e((string)$u['full_name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>

                    <div class="split">
                        <div>
                            <label>Yeni şifre</label>
                            <input name="new_password" type="password" required>
                        </div>
                        <div>
                            <label>Yeni şifre (tekrar)</label>
                            <input name="new_password2" type="password" required>
                        </div>
                    </div>

                    <div style="margin-top:14px" class="actions">
                        <button class="btn secondary" type="submit">ŞİFREYİ GÜNCELLE</button>
                    </div>
                </form>
            </div>
        </div>

        <div class="card">
            <h2>Kullanıcı Güncelle</h2>
            <div class="body">
                <form method="post" style="margin-bottom:14px">
                    <input type="hidden" name="csrf" value="<?php echo e(csrf_token()); ?>">
                    <input type="hidden" name="action" value="update_user">

                    <label>Kullanıcı</label>
                    <select name="user_id" required>
                        <option value="">Seçin...</option>
                        <?php foreach ($users as $u): ?>
                            <option value="<?php echo (int)$u['id']; ?>">
                                <?php echo e((string)$u['username']); ?> — <?php echo e((string)$u['full_name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>

                    <label>Ad Soyad *</label>
                    <input name="full_name" required>

                    <div class="split">
                        <div>
                            <label>Rol</label>
                            <select name="role" required>
                                <option value="technician">technician</option>
                                <option value="admin">admin</option>
                            </select>
                        </div>
                        <div>
                            <label>Telefon</label>
                            <input name="phone">
                        </div>
                    </div>

                    <label>E-posta</label>
                    <input name="email" type="email">

                    <div style="margin-top:14px" class="actions">
                        <button class="btn" type="submit">BİLGİLERİ GÜNCELLE</button>
                    </div>
                </form>

                <form method="post">
                    <input type="hidden" name="csrf" value="<?php echo e(csrf_token()); ?>">
                    <input type="hidden" name="action" value="toggle_active">

                    <label>Aktif/Pasif</label>
                    <div class="split">
                        <div>
                            <select name="user_id" required>
                                <option value="">Seçin...</option>
                                <?php foreach ($users as $u): ?>
                                    <option value="<?php echo (int)$u['id']; ?>">
                                        <?php echo e((string)$u['username']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div>
                            <select name="is_active" required>
                                <option value="1">aktif</option>
                                <option value="0">pasif</option>
                            </select>
                        </div>
                    </div>

                    <div style="margin-top:14px" class="actions">
                        <button class="btn danger" type="submit">DURUMU UYGULA</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
<?php
$html = ob_get_clean();
render_layout('Kullanıcı Yönetimi', $html);

