<?php
declare(strict_types=1);

require_once __DIR__ . '/../api/config/config.php';
require_once __DIR__ . '/../api/core/Database.php';

session_start();

// Basit hardening
ini_set('session.cookie_httponly', '1');
if (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') {
    ini_set('session.cookie_secure', '1');
}

function db(): PDO
{
    static $pdo = null;
    if ($pdo instanceof PDO) {
        return $pdo;
    }
    $database = new Database();
    $pdo = $database->getConnection();
    return $pdo;
}

function e(string $value): string
{
    return htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
}

function csrf_token(): string
{
    if (empty($_SESSION['csrf'])) {
        $_SESSION['csrf'] = bin2hex(random_bytes(32));
    }
    return (string)$_SESSION['csrf'];
}

function csrf_require(): void
{
    $token = $_POST['csrf'] ?? '';
    if (!is_string($token) || $token === '' || !hash_equals(csrf_token(), $token)) {
        http_response_code(400);
        echo 'Geçersiz istek (CSRF).';
        exit();
    }
}

function flash_set(string $type, string $message): void
{
    $_SESSION['flash'] = ['type' => $type, 'message' => $message];
}

function flash_get(): ?array
{
    $flash = $_SESSION['flash'] ?? null;
    unset($_SESSION['flash']);
    return is_array($flash) ? $flash : null;
}

function is_admin_logged_in(): bool
{
    return !empty($_SESSION['admin_user_id']);
}

function require_admin(): void
{
    if (!is_admin_logged_in()) {
        header('Location: login.php');
        exit();
    }

    // Session'a güvenmeden DB'den kontrol (rol/pasif kullanıcı durumu)
    $id = (int)($_SESSION['admin_user_id'] ?? 0);
    if ($id <= 0) {
        header('Location: login.php');
        exit();
    }

    try {
        $stmt = db()->prepare("SELECT role, is_active, username FROM users WHERE id = :id LIMIT 1");
        $stmt->execute([':id' => $id]);
        $u = $stmt->fetch(PDO::FETCH_ASSOC);
        $ok = is_array($u) && ($u['is_active'] ?? 0) == 1 && ($u['role'] ?? '') === 'admin';
        if (!$ok) {
            $_SESSION = [];
            session_destroy();
            header('Location: login.php');
            exit();
        }
        $_SESSION['admin_username'] = (string)($u['username'] ?? '');
    } catch (Throwable $e) {
        // DB erişilemiyorsa güvenli tarafta kal
        header('Location: login.php');
        exit();
    }
}

function current_admin_username(): string
{
    return (string)($_SESSION['admin_username'] ?? '');
}
