<?php
// Servis İşlemi Güncelleme (Admin)

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../core/Database.php';
require_once __DIR__ . '/../core/JWT.php';

$database = new Database();
$db = $database->getConnection();

// Token kontrolü
$token = JWT::getBearerToken();
if (!$token) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Token bulunamadı']);
    exit();
}

$decoded = JWT::decode($token);
if (!$decoded) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Geçersiz token']);
    exit();
}

// Admin kontrolü
if ($decoded['role'] !== 'admin') {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Yetkisiz işlem']);
    exit();
}

$data = json_decode(file_get_contents("php://input"));

if (empty($data->item_id)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'İşlem ID gerekli'
    ]);
    exit();
}

try {
    $db->beginTransaction();

    // İşlem kontrolü ve service_record_id al
    $check_query = "SELECT service_record_id FROM service_items WHERE id = :id LIMIT 1";
    $check_stmt = $db->prepare($check_query);
    $check_stmt->bindParam(':id', $data->item_id);
    $check_stmt->execute();

    if ($check_stmt->rowCount() == 0) {
        throw new Exception('İşlem bulunamadı');
    }

    $item = $check_stmt->fetch(PDO::FETCH_ASSOC);
    $service_record_id = $item['service_record_id'];

    // Güncelleme alanları
    $updates = [];
    $params = [':id' => $data->item_id];

    if (isset($data->item_type)) {
        $updates[] = "item_type = :item_type";
        $params[':item_type'] = $data->item_type;
    }

    if (isset($data->description)) {
        $updates[] = "description = :description";
        $params[':description'] = $data->description;
    }

    if (isset($data->quantity)) {
        $updates[] = "quantity = :quantity";
        $params[':quantity'] = $data->quantity;
    }

    if (isset($data->unit_price)) {
        $updates[] = "unit_price = :unit_price";
        $params[':unit_price'] = $data->unit_price;
    }

    // Toplam fiyatı hesapla
    if (isset($data->quantity) || isset($data->unit_price)) {
        // Mevcut değerleri al
        $current_query = "SELECT quantity, unit_price FROM service_items WHERE id = :id";
        $current_stmt = $db->prepare($current_query);
        $current_stmt->bindParam(':id', $data->item_id);
        $current_stmt->execute();
        $current = $current_stmt->fetch(PDO::FETCH_ASSOC);

        $new_quantity = isset($data->quantity) ? $data->quantity : $current['quantity'];
        $new_price = isset($data->unit_price) ? $data->unit_price : $current['unit_price'];
        $total_price = $new_quantity * $new_price;

        $updates[] = "total_price = :total_price";
        $params[':total_price'] = $total_price;
    }

    if (empty($updates)) {
        throw new Exception('Güncellenecek alan bulunamadı');
    }

    $update_query = "UPDATE service_items SET " . implode(', ', $updates) . " WHERE id = :id";
    $update_stmt = $db->prepare($update_query);
    $update_stmt->execute($params);

    // Servis kaydının toplam tutarını güncelle
    $update_total_query = "UPDATE service_records SET total_amount = (
                            SELECT COALESCE(SUM(total_price), 0) FROM service_items WHERE service_record_id = :service_record_id
                         ) WHERE id = :id";
    $update_total_stmt = $db->prepare($update_total_query);
    $update_total_stmt->bindParam(':service_record_id', $service_record_id);
    $update_total_stmt->bindParam(':id', $service_record_id);
    $update_total_stmt->execute();

    $db->commit();

    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'İşlem güncellendi'
    ]);

} catch (Exception $e) {
    $db->rollBack();
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
