<?php
// Yeni Servis Kaydı Oluşturma (Teknisyen ve Admin)

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../core/Database.php';
require_once __DIR__ . '/../core/JWT.php';

$database = new Database();
$db = $database->getConnection();

// Token kontrolü
$token = JWT::getBearerToken();
if (!$token) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Token bulunamadı']);
    exit();
}

$decoded = JWT::decode($token);
if (!$decoded) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Geçersiz token']);
    exit();
}

$data = json_decode(file_get_contents("php://input"));

if (empty($data->vehicle_id)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => 'Araç ID gerekli'
    ]);
    exit();
}

try {
    // Aynı araçta açık servis varsa yeni kayıt oluşturma (mantık: admin tamamlayana kadar devam etsin)
    $existing_stmt = $db->prepare(
        "SELECT id
         FROM service_records
         WHERE vehicle_id = :vehicle_id
           AND status IN ('pending', 'in_progress')
         ORDER BY created_at DESC, id DESC
         LIMIT 1"
    );
    $existing_stmt->bindParam(':vehicle_id', $data->vehicle_id);
    $existing_stmt->execute();

    if ($existing_stmt->rowCount() > 0) {
        $existing = $existing_stmt->fetch(PDO::FETCH_ASSOC);
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'existing' => true,
            'message' => 'Devam eden servis bulundu',
            'service_id' => $existing['id'],
        ]);
        exit();
    }

    $db->beginTransaction();

    // Araç kontrolü
    $vehicle_query = "SELECT id FROM vehicles WHERE id = :vehicle_id LIMIT 1";
    $vehicle_stmt = $db->prepare($vehicle_query);
    $vehicle_stmt->bindParam(':vehicle_id', $data->vehicle_id);
    $vehicle_stmt->execute();

    if ($vehicle_stmt->rowCount() == 0) {
        throw new Exception('Araç bulunamadı');
    }

    // Servis kaydı oluştur
    $service_query = "INSERT INTO service_records (vehicle_id, technician_id, service_date, description, status)
                     VALUES (:vehicle_id, :technician_id, :service_date, :description, :status)";

    $service_stmt = $db->prepare($service_query);
    $service_stmt->bindParam(':vehicle_id', $data->vehicle_id);
    $service_stmt->bindParam(':technician_id', $decoded['user_id']);

    $service_date = isset($data->service_date) ? $data->service_date : date('Y-m-d');
    $service_stmt->bindParam(':service_date', $service_date);

    $description = isset($data->description) ? $data->description : null;
    $service_stmt->bindParam(':description', $description);

    $status = isset($data->status) ? $data->status : 'pending';
    $service_stmt->bindParam(':status', $status);

    $service_stmt->execute();
    $service_id = $db->lastInsertId();

    $db->commit();

    http_response_code(201);
    echo json_encode([
        'success' => true,
        'message' => 'Servis kaydı oluşturuldu',
        'service_id' => $service_id
    ]);

} catch (Exception $e) {
    $db->rollBack();
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
