<?php
// QR Kod Tarama ve Bilgi Getirme (Public)
// Not: Bu endpoint müşteri tarafında kullanıldığı için fiyat bilgisi döndürmez.

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../core/Database.php';

$database = new Database();
$db = $database->getConnection();

$qr_hash = isset($_GET['hash']) ? $_GET['hash'] : null;
if (empty($qr_hash)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'QR kod hash değeri gerekli']);
    exit();
}

try {
    $query = "SELECT
                qc.id,
                qc.qr_hash,
                qc.qr_url,
                qc.is_assigned,
                v.id as vehicle_id,
                v.plate,
                v.brand,
                v.model,
                v.year,
                v.color,
                c.id as customer_id,
                c.name as customer_name,
                c.phone
              FROM qr_codes qc
              LEFT JOIN vehicles v ON v.qr_code_id = qc.id
              LEFT JOIN customers c ON c.id = v.customer_id
              WHERE qc.qr_hash = :qr_hash
              LIMIT 1";

    $stmt = $db->prepare($query);
    $stmt->bindParam(':qr_hash', $qr_hash);
    $stmt->execute();

    if ($stmt->rowCount() <= 0) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'QR kod bulunamadı']);
        exit();
    }

    $qr_data = $stmt->fetch(PDO::FETCH_ASSOC);

    $vehicle_id = $qr_data['vehicle_id'] ?? null;
    $flag_assigned = ($qr_data['is_assigned'] == 1 || $qr_data['is_assigned'] === true || $qr_data['is_assigned'] === '1');
    // Tutarlı sonuç: vehicle bağlıysa assigned kabul et
    $is_assigned = $flag_assigned || !empty($vehicle_id);

    $service_history = [];
    if (!empty($vehicle_id)) {
        $service_query = "SELECT sr.*, u.full_name as technician_name
                          FROM service_records sr
                          LEFT JOIN users u ON u.id = sr.technician_id
                          WHERE sr.vehicle_id = :vehicle_id
                          ORDER BY sr.service_date DESC, sr.created_at DESC";
        $service_stmt = $db->prepare($service_query);
        $service_stmt->bindParam(':vehicle_id', $vehicle_id);
        $service_stmt->execute();
        $service_history = $service_stmt->fetchAll(PDO::FETCH_ASSOC);

        $serviceIds = array_map(function ($row) {
            return (int)$row['id'];
        }, $service_history);

        $itemsByServiceId = [];
        if (!empty($serviceIds)) {
            $placeholders = implode(',', array_fill(0, count($serviceIds), '?'));
            $items_query = "SELECT
                              service_record_id,
                              item_type,
                              description,
                              quantity
                            FROM service_items
                            WHERE service_record_id IN ($placeholders)
                            ORDER BY created_at ASC, id ASC";

            $items_stmt = $db->prepare($items_query);
            $items_stmt->execute($serviceIds);
            $items = $items_stmt->fetchAll(PDO::FETCH_ASSOC);

            foreach ($items as $item) {
                $sid = (int)$item['service_record_id'];
                if (!isset($itemsByServiceId[$sid])) {
                    $itemsByServiceId[$sid] = [];
                }
                $itemsByServiceId[$sid][] = [
                    'item_type' => $item['item_type'],
                    'description' => $item['description'],
                    'quantity' => $item['quantity'],
                ];
            }
        }

        foreach ($service_history as &$service) {
            $sid = (int)$service['id'];
            $service['items'] = $itemsByServiceId[$sid] ?? [];
            $service['item_count'] = count($service['items']);
            unset($service['total_amount']);
        }
        unset($service);
    }

    http_response_code(200);
    echo json_encode([
        'success' => true,
        'qr_code' => [
            'id' => $qr_data['id'],
            'hash' => $qr_data['qr_hash'],
            'is_assigned' => $is_assigned
        ],
        'vehicle' => !empty($vehicle_id) ? [
            'id' => $vehicle_id,
            'plate' => $qr_data['plate'],
            'brand' => $qr_data['brand'],
            'model' => $qr_data['model'],
            'year' => $qr_data['year'],
            'color' => $qr_data['color']
        ] : null,
        'customer' => !empty($qr_data['customer_id']) ? [
            'id' => $qr_data['customer_id'],
            'name' => $qr_data['customer_name'],
            'phone' => $qr_data['phone']
        ] : null,
        'service_history' => $service_history
    ]);
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'İşlem sırasında hata oluştu']);
}

