<?php
// QR Kodu Müşteri/Araç'a Atama (Sadece Teknisyen ve Admin)

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../core/Database.php';
require_once __DIR__ . '/../core/JWT.php';

$database = new Database();
$db = $database->getConnection();

// Token kontrolü
$token = JWT::getBearerToken();
if (!$token) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Token bulunamadı']);
    exit();
}

$decoded = JWT::decode($token);
if (!$decoded) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Geçersiz token']);
    exit();
}

$payload = json_decode(file_get_contents('php://input'), true);
if (!is_array($payload)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Geçersiz istek gövdesi']);
    exit();
}

$qr_hash = trim((string)($payload['qr_hash'] ?? ''));
$customer_name = trim((string)($payload['customer_name'] ?? ''));
$customer_phone = trim((string)($payload['customer_phone'] ?? ''));
$vehicle_plate = trim((string)($payload['vehicle_plate'] ?? ''));
$vehicle_brand = trim((string)($payload['vehicle_brand'] ?? ''));
$vehicle_model = trim((string)($payload['vehicle_model'] ?? ''));

if ($qr_hash === '' || $customer_name === '' || $customer_phone === '' || $vehicle_plate === '' || $vehicle_brand === '' || $vehicle_model === '') {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Gerekli alanlar eksik']);
    exit();
}

try {
    // Plaka mükerrer kontrolü (boşluk ve büyük/küçük harf farkını yok say)
    $plate_normalized = strtoupper(preg_replace('/\s+/', '', $vehicle_plate));
    $vehicle_plate = $plate_normalized;
    $plate_stmt = $db->prepare("SELECT id, customer_id, plate FROM vehicles WHERE REPLACE(UPPER(plate), ' ', '') = :plate LIMIT 1");
    $plate_stmt->bindParam(':plate', $plate_normalized);
    $plate_stmt->execute();

    if ($plate_stmt->rowCount() > 0) {
        $existing = $plate_stmt->fetch(PDO::FETCH_ASSOC);
        http_response_code(409);
        echo json_encode([
            'success' => false,
            'message' => 'Bu plaka zaten kayıtlı',
            'data' => [
                'vehicle_id' => (int)($existing['id'] ?? 0),
                'customer_id' => (int)($existing['customer_id'] ?? 0),
                'plate' => $existing['plate'] ?? null,
            ],
        ]);
        exit();
    }

    // QR kodu kontrol et
    $qr_stmt = $db->prepare("SELECT id, is_assigned FROM qr_codes WHERE qr_hash = :qr_hash LIMIT 1");
    $qr_stmt->bindParam(':qr_hash', $qr_hash);
    $qr_stmt->execute();

    if ($qr_stmt->rowCount() === 0) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'QR kod bulunamadı']);
        exit();
    }

    $qr_code = $qr_stmt->fetch(PDO::FETCH_ASSOC);
    $qr_code_id = (int)$qr_code['id'];
    $flag_assigned = ($qr_code['is_assigned'] == 1 || $qr_code['is_assigned'] === true || $qr_code['is_assigned'] === '1');

    // vehicles tarafında zaten bağlı mı?
    $v_stmt = $db->prepare("SELECT id FROM vehicles WHERE qr_code_id = :qr_code_id LIMIT 1");
    $v_stmt->bindParam(':qr_code_id', $qr_code_id, PDO::PARAM_INT);
    $v_stmt->execute();
    $vehicle_exists = $v_stmt->rowCount() > 0;

    if ($flag_assigned || $vehicle_exists) {
        // Tutarsızlık varsa düzelt: vehicles var ama flag 0 ise 1 yap
        if (!$flag_assigned && $vehicle_exists) {
            $fix_stmt = $db->prepare("UPDATE qr_codes SET is_assigned = 1 WHERE id = :id");
            $fix_stmt->bindParam(':id', $qr_code_id, PDO::PARAM_INT);
            $fix_stmt->execute();
        }

        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Bu QR kod zaten atanmış']);
        exit();
    }

    $db->beginTransaction();

    // Müşteri ekle (opsiyonel alanlar geriye dönük uyumluluk için kabul edilir)
    $email = isset($payload['customer_email']) ? $payload['customer_email'] : null;
    $address = isset($payload['customer_address']) ? $payload['customer_address'] : null;

    $customer_stmt = $db->prepare("INSERT INTO customers (name, phone, email, address) VALUES (:name, :phone, :email, :address)");
    $customer_stmt->bindParam(':name', $customer_name);
    $customer_stmt->bindParam(':phone', $customer_phone);
    $customer_stmt->bindParam(':email', $email);
    $customer_stmt->bindParam(':address', $address);
    $customer_stmt->execute();
    $customer_id = (int)$db->lastInsertId();

    // Araç ekle (opsiyonel alanlar geriye dönük uyumluluk için kabul edilir)
    $year = isset($payload['vehicle_year']) ? $payload['vehicle_year'] : null;
    $color = isset($payload['vehicle_color']) ? $payload['vehicle_color'] : null;
    $chassis = isset($payload['chassis_number']) ? $payload['chassis_number'] : null;

    $vehicle_stmt = $db->prepare(
        "INSERT INTO vehicles (customer_id, qr_code_id, plate, brand, model, year, color, chassis_number)
         VALUES (:customer_id, :qr_code_id, :plate, :brand, :model, :year, :color, :chassis_number)"
    );
    $vehicle_stmt->bindParam(':customer_id', $customer_id, PDO::PARAM_INT);
    $vehicle_stmt->bindParam(':qr_code_id', $qr_code_id, PDO::PARAM_INT);
    $vehicle_stmt->bindParam(':plate', $vehicle_plate);
    $vehicle_stmt->bindParam(':brand', $vehicle_brand);
    $vehicle_stmt->bindParam(':model', $vehicle_model);
    $vehicle_stmt->bindParam(':year', $year);
    $vehicle_stmt->bindParam(':color', $color);
    $vehicle_stmt->bindParam(':chassis_number', $chassis);
    $vehicle_stmt->execute();
    $vehicle_id = (int)$db->lastInsertId();

    // QR kodu güncelle
    $update_stmt = $db->prepare("UPDATE qr_codes SET is_assigned = 1 WHERE id = :id");
    $update_stmt->bindParam(':id', $qr_code_id, PDO::PARAM_INT);
    $update_stmt->execute();

    $db->commit();

    http_response_code(201);
    echo json_encode([
        'success' => true,
        'message' => 'QR kod başarıyla atandı',
        'data' => [
            'customer_id' => $customer_id,
            'vehicle_id' => $vehicle_id,
            'qr_code_id' => $qr_code_id
        ]
    ]);
} catch (PDOException $e) {
    if ($db->inTransaction()) {
        $db->rollBack();
    }

    $msg = $e->getMessage();
    if (stripos($msg, 'Duplicate entry') !== false && stripos($msg, 'plate') !== false) {
        http_response_code(409);
        echo json_encode(['success' => false, 'message' => 'Bu plaka zaten kayıtlı']);
        exit();
    }
    if (stripos($msg, 'Duplicate entry') !== false && stripos($msg, 'qr_code_id') !== false) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Bu QR kod zaten atanmış']);
        exit();
    }

    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'İşlem sırasında hata oluştu']);
} catch (Exception $e) {
    if ($db->inTransaction()) {
        $db->rollBack();
    }
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
