<?php
// Müşteri Detayı (Admin ve Teknisyen)

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../core/Database.php';
require_once __DIR__ . '/../core/JWT.php';

$database = new Database();
$db = $database->getConnection();

// Token kontrolü
$token = JWT::getBearerToken();
if (!$token) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Token bulunamadı']);
    exit();
}

$decoded = JWT::decode($token);
if (!$decoded) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Geçersiz token']);
    exit();
}

if (empty($_GET['id'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Müşteri ID gerekli']);
    exit();
}

try {
    $customer_id = $_GET['id'];

    // Müşteri bilgileri
    $customer_query = "SELECT * FROM customers WHERE id = :id LIMIT 1";
    $customer_stmt = $db->prepare($customer_query);
    $customer_stmt->bindParam(':id', $customer_id);
    $customer_stmt->execute();

    if ($customer_stmt->rowCount() == 0) {
        throw new Exception('Müşteri bulunamadı');
    }

    $customer = $customer_stmt->fetch(PDO::FETCH_ASSOC);

    // Araçlar
    $vehicles_query = "SELECT * FROM vehicles WHERE customer_id = :customer_id ORDER BY plate ASC";
    $vehicles_stmt = $db->prepare($vehicles_query);
    $vehicles_stmt->bindParam(':customer_id', $customer_id);
    $vehicles_stmt->execute();
    $vehicles = $vehicles_stmt->fetchAll(PDO::FETCH_ASSOC);

    // Servis geçmişi
    $services_query = "SELECT
                        sr.*,
                        v.plate,
                        v.brand,
                        v.model,
                        u.full_name as technician_name
                      FROM service_records sr
                      JOIN vehicles v ON v.id = sr.vehicle_id
                      JOIN users u ON u.id = sr.technician_id
                      WHERE v.customer_id = :customer_id
                      ORDER BY sr.service_date DESC, sr.created_at DESC";
    $services_stmt = $db->prepare($services_query);
    $services_stmt->bindParam(':customer_id', $customer_id);
    $services_stmt->execute();
    $services = $services_stmt->fetchAll(PDO::FETCH_ASSOC);

    // Ödemeler
    $payments_query = "SELECT
                        p.*,
                        u.full_name as created_by_name,
                        sr.id as service_id,
                        v.plate
                      FROM payments p
                      LEFT JOIN users u ON u.id = p.created_by
                      LEFT JOIN service_records sr ON sr.id = p.service_record_id
                      LEFT JOIN vehicles v ON v.id = sr.vehicle_id
                      WHERE p.customer_id = :customer_id
                      ORDER BY p.payment_date DESC, p.created_at DESC";
    $payments_stmt = $db->prepare($payments_query);
    $payments_stmt->bindParam(':customer_id', $customer_id);
    $payments_stmt->execute();
    $payments = $payments_stmt->fetchAll(PDO::FETCH_ASSOC);

    // Finansal özet
    $summary_query = "SELECT
                        (SELECT COALESCE(SUM(sr.total_amount), 0)
                         FROM service_records sr
                         JOIN vehicles v ON v.id = sr.vehicle_id
                         WHERE v.customer_id = :customer_id1) as total_debt,
                        (SELECT COALESCE(SUM(p.amount), 0)
                         FROM payments p
                         WHERE p.customer_id = :customer_id2) as total_paid";
    $summary_stmt = $db->prepare($summary_query);
    $summary_stmt->bindParam(':customer_id1', $customer_id);
    $summary_stmt->bindParam(':customer_id2', $customer_id);
    $summary_stmt->execute();
    $summary_row = $summary_stmt->fetch(PDO::FETCH_ASSOC);

    $summary = [
        'total_debt' => $summary_row['total_debt'],
        'total_paid' => $summary_row['total_paid'],
        'balance' => $summary_row['total_debt'] - $summary_row['total_paid']
    ];

    http_response_code(200);
    echo json_encode([
        'success' => true,
        'customer' => $customer,
        'vehicles' => $vehicles,
        'services' => $services,
        'payments' => $payments,
        'summary' => $summary
    ]);

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
