<?php
// Admin Raporları (Haftalık, Aylık) (Sadece Admin)

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../core/Database.php';
require_once __DIR__ . '/../core/JWT.php';

$database = new Database();
$db = $database->getConnection();

// Token kontrolü
$token = JWT::getBearerToken();
if (!$token) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Token bulunamadı']);
    exit();
}

$decoded = JWT::decode($token);
if (!$decoded || $decoded['role'] != 'admin') {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Yetkiniz yok']);
    exit();
}

$report_type = isset($_GET['type']) ? $_GET['type'] : 'weekly';

$start_date = '';
$end_date = date('Y-m-d');

if ($report_type == 'weekly') {
    $start_date = date('Y-m-d', strtotime('-7 days'));
} else if ($report_type == 'monthly') {
    $start_date = date('Y-m-d', strtotime('-30 days'));
} else if ($report_type == 'yearly') {
    $start_date = date('Y-m-d', strtotime('-365 days'));
}

// Toplam gelir
$income_query = "SELECT COALESCE(SUM(amount), 0) as total_income
                FROM payments
                WHERE payment_date BETWEEN :start_date AND :end_date";
$income_stmt = $db->prepare($income_query);
$income_stmt->bindParam(':start_date', $start_date);
$income_stmt->bindParam(':end_date', $end_date);
$income_stmt->execute();
$income = $income_stmt->fetch();

// Toplam servis sayısı
$service_query = "SELECT COUNT(*) as total_services
                 FROM service_records
                 WHERE service_date BETWEEN :start_date AND :end_date";
$service_stmt = $db->prepare($service_query);
$service_stmt->bindParam(':start_date', $start_date);
$service_stmt->bindParam(':end_date', $end_date);
$service_stmt->execute();
$services = $service_stmt->fetch();

// Toplam borç (tüm zamanlar - şu anki genel bakiye)
$debt_query = "SELECT
                COALESCE(SUM(GREATEST(COALESCE(d.total_debt, 0) - COALESCE(p.total_paid, 0), 0)), 0) as total_debt
              FROM customers c
              LEFT JOIN (
                SELECT v.customer_id, SUM(sr.total_amount) AS total_debt
                FROM vehicles v
                JOIN service_records sr ON sr.vehicle_id = v.id
                GROUP BY v.customer_id
              ) d ON d.customer_id = c.id
              LEFT JOIN (
                SELECT customer_id, SUM(amount) AS total_paid
                FROM payments
                GROUP BY customer_id
              ) p ON p.customer_id = c.id";
$debt_stmt = $db->prepare($debt_query);
$debt_stmt->execute();
$debt = $debt_stmt->fetch();

// En çok borcu olan müşteriler
$top_debtors_query = "SELECT
                        c.id AS customer_id,
                        c.name AS customer_name,
                        c.phone,
                        COALESCE(d.total_debt, 0) AS total_debt,
                        COALESCE(p.total_paid, 0) AS total_paid,
                        COALESCE(d.total_debt, 0) - COALESCE(p.total_paid, 0) AS balance
                      FROM customers c
                      LEFT JOIN (
                        SELECT v.customer_id, SUM(sr.total_amount) AS total_debt
                        FROM vehicles v
                        JOIN service_records sr ON sr.vehicle_id = v.id
                        GROUP BY v.customer_id
                      ) d ON d.customer_id = c.id
                      LEFT JOIN (
                        SELECT customer_id, SUM(amount) AS total_paid
                        FROM payments
                        GROUP BY customer_id
                      ) p ON p.customer_id = c.id
                      WHERE (COALESCE(d.total_debt, 0) - COALESCE(p.total_paid, 0)) > 0
                      ORDER BY balance DESC
                      LIMIT 10";
$top_debtors_stmt = $db->prepare($top_debtors_query);
$top_debtors_stmt->execute();
$top_debtors = $top_debtors_stmt->fetchAll();

// Günlük ödemeler
$daily_payments_query = "SELECT payment_date, SUM(amount) as daily_total
                        FROM payments
                        WHERE payment_date BETWEEN :start_date AND :end_date
                        GROUP BY payment_date
                        ORDER BY payment_date DESC";
$daily_payments_stmt = $db->prepare($daily_payments_query);
$daily_payments_stmt->bindParam(':start_date', $start_date);
$daily_payments_stmt->bindParam(':end_date', $end_date);
$daily_payments_stmt->execute();
$daily_payments = $daily_payments_stmt->fetchAll();

http_response_code(200);
echo json_encode([
    'success' => true,
    'report' => [
        'type' => $report_type,
        'start_date' => $start_date,
        'end_date' => $end_date,
        'total_income' => $income['total_income'],
        'total_services' => $services['total_services'],
        'total_debt' => $debt['total_debt'],
        'outstanding_balance' => $debt['total_debt'],
        'top_debtors' => $top_debtors,
        'daily_payments' => $daily_payments
    ]
]);
